function [ROIs] = SPM8_Centaur(inT1,inPET,tracer,optOutDir)
% Implements the Centiloid Standard SPM8 pipeline from Klunk et al. https://doi.org/10.1016/j.jalz.2014.07.003 with the
% CenTauR ROIs and CenTaurZ constants from https://doi.org/10.1002/alz.061177. 
% 
% Outputs values in SUVR units for each CenTauR ROI, and also in CenTaurZ units.
% 
% Supported tracers are {'RO948','FTP','MK6240','GTP1','PM-PBB3','PI2620'}.
%
% Inputs must be summed, late-uptake tau PET images and T1-w MRI in .nii format. 
%
% Requires SPM8 
% Also requires the included ROIs and csv files to be in your matlab path
%
% Copyright 2024 Mayo Clinic. 
% Contact: schwarz.christopher@mayo.edu

% addpath(genpath('./SPM8'))
addpath('./ROIs/CenTauR/')

%% preamble
[~,t1b,]=fileparts(inT1);
[~,petb,]=fileparts(inPET);

if( exist('optOutDir','var') && ~isempty(optOutDir) )
    if(~isfolder(optOutDir))
        mkdir(optOutDir)
    end
    outDir = optOutDir;
else
    outDir = pwd;
end

ROIs(1).name = 'Universal';
ROIs(1).mask = which('CenTauR.nii');
ROIs(2).name = 'MesialTemporal';
ROIs(2).mask = which('Mesial_CenTauR.nii');
ROIs(3).name = 'MetaTemporal';
ROIs(3).mask = which('Meta_CenTauR.nii');
ROIs(4).name = 'TemporoParietal';
ROIs(4).mask = which('TP_CenTauR.nii');
ROIs(5).name = 'Frontal';
ROIs(5).mask = which('Frontal_CenTauR.nii');

for i=1:numel(ROIs)
    if(isempty(ROIs(i).mask))
        error(['Could not find mask file for: ' ROIs(i).name])
    end
    ROIs(i).hdr = spm_vol(ROIs(i).mask);
    ROIs(i).vol = spm_read_vols(ROIs(i).hdr);
end

constant_table = which('centaur_constants.csv');
if(isempty(constant_table))
    error('Centaur table of constants was not found in your matlab path.')
end
c = table2struct(readtable(constant_table));

ref_mask = which('voi_CerebGry_tau_2mm.nii');
if(isempty(ref_mask))
    error('Could not find reference mask file');
end
roi_ref_hdr = spm_vol(ref_mask);
roi_ref_vol = spm_read_vols(roi_ref_hdr);

supported_tracers = {'RO948','FTP','MK6240','GTP1','PM-PBB3','PI2620'};
if(~ismember(tracer,supported_tracers))
    error(['''' tracer '''' ' is not a supported tracer. You must input one of: ' strjoin(supported_tracers,',')]);
end

copyfile(inT1,outDir);
copyfile(inPET,outDir);

owd = pwd;
cd(outDir);
cleanupVar = onCleanup(@()cd(owd));

t1 = fullfile(pwd,[t1b '.nii']);
pet = fullfile(pwd,[petb '.nii']);
t1_coreg = [pwd filesep t1b '_coregMNI.nii'];
pet_coreg = [pwd filesep petb '_coregT1MNI.nii'];

disp('Running SPM8 Centaur pipeline');

%% coreg T1 to MNI
mni = which('spm8/templates/T1.nii');
if(isfile(t1_coreg))
    disp('Found T1 coreg file. We will use it.');
else
    disp('Doing coreg T1 to MNI');
    copyfile(t1,t1_coreg);
    matlabbatch{1}.spm.spatial.coreg.estimate.ref = {[mni ',1']};
    matlabbatch{1}.spm.spatial.coreg.estimate.source = {[t1_coreg ',1']};
    matlabbatch{1}.spm.spatial.coreg.estimate.other = {''};
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.cost_fun = 'nmi';
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.sep = [4 2];
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.tol = [0.02 0.02 0.02 0.001 0.001 0.001 0.01 0.01 0.01 0.001 0.001 0.001];
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.fwhm = [7 7];
    spm_jobman('run', matlabbatch);
    clear matlabbatch
end

%% coreg PET to T1
if(isfile(pet_coreg))
    disp('Found PET coreg file. We will use it.');
else
    disp('Doing coreg PET to T1');
    copyfile(pet,pet_coreg);
    matlabbatch{1}.spm.spatial.coreg.estimate.ref = {[t1_coreg ',1']};
    matlabbatch{1}.spm.spatial.coreg.estimate.source = {[pet_coreg ',1']};
    matlabbatch{1}.spm.spatial.coreg.estimate.other = {''};
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.cost_fun = 'nmi';
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.sep = [4 2];
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.tol = [0.02 0.02 0.02 0.001 0.001 0.001 0.01 0.01 0.01 0.001 0.001 0.001];
    matlabbatch{1}.spm.spatial.coreg.estimate.eoptions.fwhm = [7 7];
    spm_jobman('run', matlabbatch);
    clear matlabbatch
end



%% segment T1
sn_mat = [pwd filesep t1b '_coregMNI_seg_sn.mat'];
if(isfile(sn_mat))
    disp('Found T1 segmentation files. We will use them.');
else
disp('Segmenting T1')
    matlabbatch{1}.spm.spatial.preproc.data = {[t1_coreg ',1']};
    matlabbatch{1}.spm.spatial.preproc.output.GM = [0 0 1];
    matlabbatch{1}.spm.spatial.preproc.output.WM = [0 0 1];
    matlabbatch{1}.spm.spatial.preproc.output.CSF = [0 0 0];
    matlabbatch{1}.spm.spatial.preproc.output.biascor = 1;
    matlabbatch{1}.spm.spatial.preproc.output.cleanup = 0;
    matlabbatch{1}.spm.spatial.preproc.opts.tpm = {
                                                   which('tpm/grey.nii')
                                                   which('tpm/white.nii')
                                                   which('tpm/csf.nii')
                                                   };
    matlabbatch{1}.spm.spatial.preproc.opts.ngaus = [2
                                                     2
                                                     2
                                                     4];
    matlabbatch{1}.spm.spatial.preproc.opts.regtype = 'mni';
    matlabbatch{1}.spm.spatial.preproc.opts.warpreg = 1;
    matlabbatch{1}.spm.spatial.preproc.opts.warpco = 25;
    matlabbatch{1}.spm.spatial.preproc.opts.biasreg = 0.0001;
    matlabbatch{1}.spm.spatial.preproc.opts.biasfwhm = 60;
    matlabbatch{1}.spm.spatial.preproc.opts.samp = 3;
    matlabbatch{1}.spm.spatial.preproc.opts.msk = {''};
    spm_jobman('run', matlabbatch);
    clear matlabbatch
end

%% Write normalized to MNI
wT1 = [pwd filesep 'w' t1b '_coregMNI.nii'];
wPET = [pwd filesep 'w' petb '_coregT1MNI.nii'];
if(isfile(wT1) && isfile(wPET))
    disp('Found warped T1 and Warped PET. We will use them.');
else
    matlabbatch{1}.spm.spatial.normalise.write.subj.matname = {sn_mat};
    matlabbatch{1}.spm.spatial.normalise.write.subj.resample = {
                                                                [t1_coreg ',1']
                                                                [pet_coreg ',1']
                                                                };
    matlabbatch{1}.spm.spatial.normalise.write.roptions.preserve = 0;
    matlabbatch{1}.spm.spatial.normalise.write.roptions.bb = [-90 -126 -72 % From the Centiloid paper, not SPM default
                                                              91 91 109];
    matlabbatch{1}.spm.spatial.normalise.write.roptions.vox = [2 2 2];
    matlabbatch{1}.spm.spatial.normalise.write.roptions.interp = 1;
    matlabbatch{1}.spm.spatial.normalise.write.roptions.wrap = [0 0 0];
    matlabbatch{1}.spm.spatial.normalise.write.roptions.prefix = 'w';
    spm_jobman('run', matlabbatch);
    clear matlabbatch
end

%% Quantify
disp('Quantifying');

wPET_hdr = spm_vol(wPET);
wPET_vol = spm_read_vols(wPET_hdr);
roi_ref_mean = mean(wPET_vol(roi_ref_vol==1));

CentaurZ_c = c(strcmp({c.Units},'CentaurZ') & strcmp({c.Tracer},tracer));
for i=1:numel(ROIs)
    ROIs(i).mean = mean(wPET_vol(ROIs(i).vol==1));
    ROIs(i).SUVR = ROIs(i).mean / roi_ref_mean;
    ROIs(i).CentaurZ = ROIs(i).SUVR * CentaurZ_c.([ROIs(i).name '_slope']) + CentaurZ_c.([ROIs(i).name '_inter']);
end
ROIs = rmfield(ROIs,'hdr');
ROIs = rmfield(ROIs,'vol');
ROIs = rmfield(ROIs,'mask');
ROIs = rmfield(ROIs,'mean');
ROIs = struct2table(ROIs);

end
