function [ROIs] = SPM12_Centaur(inT1,inPET,tracer,optOutDir)
% A pipeline to calculate SUVR and CenTaurZ values for arbitrary tau PET images, using SPM12 and MCALT.
% 
% Uses CenTauR ROIs and CenTaurZ constants from https://doi.org/10.1002/alz.061177. 
% 
% Supported tracers are {'RO948','FTP','MK6240','GTP1','PM-PBB3','PI2620'}. 
% 
% Inputs must be summed, late-uptake tau PET images and T1-w MRI in .nii format.
%
% Requires SPM12 
% Also requires the included ROIs and csv files to be in your matlab path
%
% Copyright 2024 Mayo Clinic. 
% Contact: schwarz.christopher@mayo.edu

% addpath(genpath('./SPM12'))
addpath('./ROIs/CenTauR_to_MCALT/')

%% preamble
[~,t1b,]=fileparts(inT1);
[~,petb,]=fileparts(inPET);

if( exist('optOutDir','var') && ~isempty(optOutDir) )
    if(~isfolder(optOutDir))
        mkdir(optOutDir)
    end
    outDir = optOutDir;
else
    outDir = pwd;
end

ROIs(1).name = 'Universal';
ROIs(1).mask = which('MCALT_CenTauR.nii');
ROIs(2).name = 'MesialTemporal';
ROIs(2).mask = which('MCALT_Mesial_CenTauR.nii');
ROIs(3).name = 'MetaTemporal';
ROIs(3).mask = which('MCALT_Meta_CenTauR.nii');
ROIs(4).name = 'TemporoParietal';
ROIs(4).mask = which('MCALT_TP_CenTauR.nii');
ROIs(5).name = 'Frontal';
ROIs(5).mask = which('MCALT_Frontal_CenTauR.nii');

centaur_constants = which('centaur_constants.csv');
if(isempty(centaur_constants))
    error('Centaur table of constants was not found in your matlab path.')
end
c = table2struct(readtable(centaur_constants));

pipeline_constants = which('Pipeline_constants_SPM12.csv');
if(isempty(pipeline_constants))
    error('Pipeline table of constants was not found in your matlab path.')
end
cPipe = table2struct(readtable(pipeline_constants));



ref_mask = which('MCALT_voi_CerebGry_tau_2mm.nii');
if(isempty(ref_mask))
    error('Could not find reference mask file');
end

MCALT_tpm = which('MCALT_tpm.nii');
if(isempty(MCALT_tpm))
    error('Could not find MCALT_tpm.nii');
end

MCALT_T1 = which('MCALT_T1.nii');
if(isempty(MCALT_T1))
    error('Could not find MCALT_T1.nii');
end

supported_tracers = {'RO948','FTP','MK6240','GTP1','PM-PBB3','PI2620'};
if(~ismember(tracer,supported_tracers))
    error(['''' tracer '''' ' is not a supported tracer. You must input one of: ' strjoin(supported_tracers,',')]);
end

copyfile(inT1,outDir);
copyfile(inPET,outDir);

owd = pwd;
cd(outDir);
cleanupVar = onCleanup(@()cd(owd));

t1 = fullfile(pwd,[t1b '.nii']);
pet = fullfile(pwd,[petb '.nii']);
pet_coreg = [pwd filesep petb '_coregT1.nii'];

disp('Running SPM12 Centaur pipeline');

%% coreg PET to T1 (and reslice it)
rpet_coreg = [pwd filesep 'r' petb '_coregT1.nii'];
if(isfile(rpet_coreg))
    disp('Found PET in T1 space. We will use it.');
else
    disp('Doing coreg PET to T1');
    copyfile(pet,pet_coreg);

    matlabbatch{1}.spm.spatial.coreg.estwrite.ref = {[t1 ',1']};
    matlabbatch{1}.spm.spatial.coreg.estwrite.source = {[pet_coreg ',1']};
    matlabbatch{1}.spm.spatial.coreg.estwrite.other = {''};
    matlabbatch{1}.spm.spatial.coreg.estwrite.eoptions.cost_fun = 'nmi';
    matlabbatch{1}.spm.spatial.coreg.estwrite.eoptions.sep = [4 2];
    matlabbatch{1}.spm.spatial.coreg.estwrite.eoptions.tol = [0.02 0.02 0.02 0.001 0.001 0.001 0.01 0.01 0.01 0.001 0.001 0.001];
    matlabbatch{1}.spm.spatial.coreg.estwrite.eoptions.fwhm = [7 7];
    matlabbatch{1}.spm.spatial.coreg.estwrite.roptions.interp = 4; %b-spline
    matlabbatch{1}.spm.spatial.coreg.estwrite.roptions.wrap = [0 0 0];
    matlabbatch{1}.spm.spatial.coreg.estwrite.roptions.mask = 0;
    matlabbatch{1}.spm.spatial.coreg.estwrite.roptions.prefix = 'r';

    spm_jobman('run', matlabbatch);
    clear matlabbatch
end


%% segment T1
sn_mat = [pwd filesep 'iy_' t1b '.nii'];
if(isfile(sn_mat))
    disp('Found T1 segmentation files. We will use them.');
else
    disp('Segmenting T1')
    %-----------------------------------------------------------------------
    matlabbatch{1}.spm.spatial.preproc.channel.vols = {t1};
    matlabbatch{1}.spm.spatial.preproc.channel.biasreg = 0.001;
    matlabbatch{1}.spm.spatial.preproc.channel.biasfwhm = 60;
    matlabbatch{1}.spm.spatial.preproc.channel.write = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(1).tpm = {[MCALT_tpm ',1']};
    matlabbatch{1}.spm.spatial.preproc.tissue(1).ngaus = 1;
    matlabbatch{1}.spm.spatial.preproc.tissue(1).native = [1 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(1).warped = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(2).tpm = {[MCALT_tpm ',2']};
    % Differs from standard MCALT, because using 2 Gaussians for WM requires patching SPM12 files
    matlabbatch{1}.spm.spatial.preproc.tissue(2).ngaus = 1;
    matlabbatch{1}.spm.spatial.preproc.tissue(2).native = [1 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(2).warped = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(3).tpm = {[MCALT_tpm ',3']};
    matlabbatch{1}.spm.spatial.preproc.tissue(3).ngaus = 2;
    matlabbatch{1}.spm.spatial.preproc.tissue(3).native = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(3).warped = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(4).tpm = {[MCALT_tpm ',4']};
    matlabbatch{1}.spm.spatial.preproc.tissue(4).ngaus = 3;
    matlabbatch{1}.spm.spatial.preproc.tissue(4).native = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(4).warped = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(5).tpm = {[MCALT_tpm ',5']};
    matlabbatch{1}.spm.spatial.preproc.tissue(5).ngaus = 4;
    matlabbatch{1}.spm.spatial.preproc.tissue(5).native = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(5).warped = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(6).tpm = {[MCALT_tpm ',6']};
    matlabbatch{1}.spm.spatial.preproc.tissue(6).ngaus = 2;
    matlabbatch{1}.spm.spatial.preproc.tissue(6).native = [0 0];
    matlabbatch{1}.spm.spatial.preproc.tissue(6).warped = [0 0];
    matlabbatch{1}.spm.spatial.preproc.warp.mrf = 1;
    matlabbatch{1}.spm.spatial.preproc.warp.cleanup = 1;
    matlabbatch{1}.spm.spatial.preproc.warp.reg = [0 0.001 0.5 0.05 0.2];
    matlabbatch{1}.spm.spatial.preproc.warp.affreg = 'none';
    matlabbatch{1}.spm.spatial.preproc.warp.fwhm = 0;
    matlabbatch{1}.spm.spatial.preproc.warp.samp = 3;
    matlabbatch{1}.spm.spatial.preproc.warp.write = [1 0]; % inverse, fwd
    matlabbatch{1}.spm.spatial.preproc.warp.vox = NaN;
    matlabbatch{1}.spm.spatial.preproc.warp.bb = [NaN NaN NaN
        NaN NaN NaN];

    disp('Initializing subject->template registration with spm_coreg');
    regInit=spm_coreg(matlabbatch{1}.spm.spatial.preproc.channel.vols{1},MCALT_T1);
    segAffine=spm_matrix(regInit);
    matlabbatch{1}.spm.spatial.preproc.warp.Affine=segAffine;
    spm_jobman('run', matlabbatch);
    clear matlabbatch
end

%% Transform ROIs to T1 space
wCentaur = [pwd filesep 'wMCALT_CenTauR.nii'];
wRef = [pwd filesep 'wMCALT_voi_CerebGry_tau_2mm.nii'];
if(isfile(wCentaur) && isfile(wRef))
    disp('Found ROIs. We will use them.');
else
    matlabbatch{1}.spm.util.defs.comp{1}.def = {sn_mat};
    matlabbatch{1}.spm.util.defs.out{1}.pull.fnames = [ {ROIs.mask}' ; ref_mask ];
    matlabbatch{1}.spm.util.defs.out{1}.pull.savedir.savepwd = 1;
    matlabbatch{1}.spm.util.defs.out{1}.pull.interp = -1;
    matlabbatch{1}.spm.util.defs.out{1}.pull.mask = 0;
    matlabbatch{1}.spm.util.defs.out{1}.pull.fwhm = [0 0 0];
    matlabbatch{1}.spm.util.defs.out{1}.pull.prefix = 'w';

    spm_jobman('run', matlabbatch);
    clear matlabbatch
end

for i=1:numel(ROIs)
    if(isempty(ROIs(i).mask))
        error(['Could not find mask file for: ' ROIs(i).name])
    end
    [~,maskb,~] = fileparts(ROIs(i).mask);
    ROIs(i).hdr = spm_vol(['w' maskb '.nii']);
    ROIs(i).vol = spm_read_vols(ROIs(i).hdr);
end
roi_ref_hdr = spm_vol(wRef);
roi_ref_vol = spm_read_vols(roi_ref_hdr);


%% Quantify
disp('Quantifying');

rPET_hdr = spm_vol(rpet_coreg);
rPET_vol = spm_read_vols(rPET_hdr);
roi_ref_mean = mean(rPET_vol(roi_ref_vol==1));

% Unlike the Centiloid standard SPM8 pipeline, we are operating in T1 image native space. We use the segmentations to
% remove voxels considered mostly CSF. This is sometimes called "tissue sharpening".

c1_hdr = spm_vol(['c1' t1b '.nii']);
c1_vol = spm_read_vols(c1_hdr);
c2_hdr = spm_vol(['c2' t1b '.nii']);
c2_vol = spm_read_vols(c2_hdr);

CentaurZ_c = c(strcmp({c.Units},'CentaurZ') & strcmp({c.Tracer},tracer));
Pipe_c = cPipe;
for i=1:numel(ROIs)
    ROIs(i).mean = mean(rPET_vol(ROIs(i).vol==1 & (c1_vol + c2_vol) >= 0.5));
    ROIs(i).SUVR = ROIs(i).mean / roi_ref_mean;
    % fprintf('  %s * %f + %f for SPM12 to SPM8, * %f + %f for SPM8 to CentaurZ (%s)\n',ROIs(i).name,Pipe_c(strcmp({Pipe_c.Region},ROIs(i).name),:).slope,Pipe_c(strcmp({Pipe_c.Region},ROIs(i).name),:).inter,CentaurZ_c.([ROIs(i).name '_slope']),CentaurZ_c.([ROIs(i).name '_inter']),tracer);
    ROIs(i).SUVR_SPM8 = ROIs(i).SUVR * Pipe_c(strcmp({Pipe_c.Region},ROIs(i).name),:).slope + Pipe_c(strcmp({Pipe_c.Region},ROIs(i).name),:).inter;
    ROIs(i).CentaurZ = ROIs(i).SUVR_SPM8 * CentaurZ_c.([ROIs(i).name '_slope']) + CentaurZ_c.([ROIs(i).name '_inter']);
end
ROIs = rmfield(ROIs,'hdr');
ROIs = rmfield(ROIs,'vol');
ROIs = rmfield(ROIs,'mask');
ROIs = rmfield(ROIs,'mean');
ROIs = struct2table(ROIs);

end
